import axios from "axios";
import https from "https";
import tls from "tls";
import 'dotenv/config';
import { constants } from "crypto";

const SAP_BASE_URL = process.env.SAP_BASE_URL;
const SAP_USERNAME = process.env.SAP_USERNAME;
const SAP_PASSWORD = process.env.SAP_PASSWORD;
const INSTALLMENT_URL = process.env.INSTALLMENT_URL;


const httpsAgent = new https.Agent({
  rejectUnauthorized: false,
  secureOptions: constants.SSL_OP_LEGACY_SERVER_CONNECT // enables legacy renegotiation
});

export const getEmailFromSAP = async (nationalId) => {
  try {
    const trimmedId = nationalId.padStart(14, " ");
    const params = {
      "$format": "json",
      "$select": "email,partnerCode",
      "$filter": `partnerId eq '${trimmedId}'`,
      "$top": 1,
      // "sap-client": "110" 

    };
    // console.log("SAP_BASE_URL:", SAP_BASE_URL);
    // console.log("Request params:", params);

    const response = await axios.get(SAP_BASE_URL, {
      auth: {
        username: SAP_USERNAME,
        password: SAP_PASSWORD
      },
      params,
      httpsAgent 
    });
    // console.log("SAP raw response:", response.data); 


    const results = response.data?.d?.results || [];
    if (results.length > 0) {
      return results[0].email;
    }
    return null;
  } catch (error) {
    console.error("SAP Error:", error.message);
    throw new Error("Failed to fetch from SAP");
  }
};



export const getPartnerCodeFromSAP = async (nationalId) => {
  try {
    const trimmedId = nationalId.padStart(14, " ");
    const params = {
      "$format": "json",
      "$select": "partnerCode",
      "$filter": `partnerId eq '${trimmedId}'`,
      "$top": 1,
      // "sap-client": "110"
    };
    const response = await axios.get(SAP_BASE_URL, {
      auth: {
        username: SAP_USERNAME,
        password: SAP_PASSWORD
      },
      params,
      httpsAgent
    });
    const results = response.data?.d?.results || [];
    if (results.length > 0) {
      return results[0].partnerCode;
    }
    return null;
  } catch (error) {
    console.error("SAP Error (partnerCode):", error.message);
    throw new Error("Failed to fetch partnerCode from SAP");
  }
};

export const getUnitsByPartnerCode = async (partnerCode) => {
  try {
    const params = {
      "$format": "json",
      "$select": "unit,soldprice,contract,Project_NAME",
      "$filter": `partnerCode eq '${partnerCode}' and meas eq 'Z100'`,
      // "sap-client": "110"
    };

    const response = await axios.get(SAP_BASE_URL, {
      auth: {
        username: SAP_USERNAME,
        password: SAP_PASSWORD
      },
      params,
      httpsAgent
    });
    const results = response.data?.d?.results || [];
    return results.map(r => ({
      unit: r.unit,
      soldprice: r.soldprice,
      installments: r.installments,
      contract: r.contract,
      project: r.Project_NAME
     }));
    // Return array of unit codes
    // return results.map(r => r.unit);
  } catch (error) {
    console.error("SAP Error (units):", error.message);
    throw new Error("Failed to fetch units from SAP");
  }
};

export const getUserInfoFromSAP = async (partnerCode) => {
  try {
    const params = {
      "$format": "json",
      "$select": "englishName",
      "$filter": `partnerCode eq '${partnerCode}'`,
      "$top": 1,
      // "sap-client": "110"
    };
    const response = await axios.get(SAP_BASE_URL, {
      auth: {
        username: SAP_USERNAME,
        password: SAP_PASSWORD
      },
      params,
      httpsAgent
    });
    // console.log("SAP user info response:", response.data); 
    const results = response.data?.d?.results || [];
    if (results.length > 0) {
      return { englishName: results[0].englishName };
    }
    return { englishName: results[0]?.englishName || null };
  } catch (error) {
    console.error("SAP Error (userInfo):", error.message);
    throw new Error("Failed to fetch user info from SAP");
  }
};

export const getInstallmentsByContractNo = async (contractNo) => {
  try {
    const paddedContractNo = `0000${contractNo}`;
    const params = {
      "$format": "json",
      "$filter": `contract_no eq '${paddedContractNo}'`,
      // "sap-client": "110"
    };
    const response = await axios.get(INSTALLMENT_URL, {
      auth: {
        username: SAP_USERNAME,
        password: SAP_PASSWORD
      },
      params,
      httpsAgent
    });
    const results = response.data?.d?.results || [];
    return results.map(r => ({
      dueDate: r.cond_due_date_str,
      amount: r.cond_amount,
      type: r.cond_type_text,
      status: Number(r.paid_amount) >= Number(r.cond_amount) ? "Paid" : "Unpaid",
      payRef: r.pey_ref
    }));
  } catch (error) {
    console.error("SAP Error (installments):", error.message);
    throw new Error("Failed to fetch installments from SAP");
  }
};